/*
* cam_fs_wrapper.c - Sigmastar
*
* Copyright (C) 2018 Sigmastar Technology Corp.
*
* Author: giggs.huang <giggs.huang@sigmastar.com.tw>
*
* This software is licensed under the terms of the GNU General Public
* License version 2, as published by the Free Software Foundation, and
* may be copied, distributed, and modified under those terms.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
*/


///////////////////////////////////////////////////////////////////////////////
/// @file      cam_fs_wrapper.c
/// @brief     Cam FS Wrapper Source File for
///            1. RTK OS
///            2. Linux User Space
///            3. Linux Kernel Space
///////////////////////////////////////////////////////////////////////////////

#if defined(__KERNEL__)
#define CAM_OS_LINUX_KERNEL
#endif

#ifdef CAM_OS_RTK
#include "sys_sys.h"
#include "cam_os_wrapper.h"
#include "cam_fs_wrapper.h"

int DrvSpinandProbe(void);
U32 MDrv_SPINAND_LoadWB(U8* pu8DMAAddr, U32 u32DataSize,  U16 u16_PartType);
U32 MDrv_SPINAND_LoadBL(U8* pu8DMAAddr, U32 u32_BLSize, bool b_InternalECC, U16 u16_PartType);
U32 DrvSpinand_ReadBlockPba(U8* pu8DmaAddr, U32 u32Pba, U32 u32DataSize);

typedef struct
{
    U8   u8Mid;
    U32  u32Did;
        #define DEFAULT_DID ((U32)-1) //any device ID
    U32 (*EnableOtp)(bool bEnable);
} SPINAND_Hal_t;

SPINAND_Hal_t tHal;

#elif defined(CAM_OS_LINUX_USER)
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include "cam_os_wrapper.h"
#include "cam_fs_wrapper.h"

#elif defined(CAM_OS_LINUX_KERNEL)
#include <linux/module.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/unistd.h>
#include <linux/sched.h>
#include <linux/fs.h>
#include <linux/file.h>
#include <linux/mm.h>
#include <asm/uaccess.h>
#include "cam_os_wrapper.h"
#include "cam_fs_wrapper.h"
#endif

CamFsRet_e CamFsOpen(CamFsFd *ptFd, const char *szPath, u32 nFlag, u32 nMode)
{
#ifdef CAM_OS_RTK
    /*
      There is no file system in RTK now, CamFs will access spinand device
      directly. CamFsOpen will use PBA number instead file descriptor.
    */
    char *pEnd = NULL;

    *ptFd = (CamFsFd)strtoul(szPath, &pEnd, 10);

    return (szPath == pEnd)? CAM_FS_FAIL : CAM_FS_OK;
#elif defined(CAM_OS_LINUX_USER)
    return CAM_FS_FAIL;
#elif defined(CAM_OS_LINUX_KERNEL)
    struct file *ptFp = NULL;
    mm_segment_t tFs;

    tFs = get_fs();
    set_fs(get_ds());
    ptFp = filp_open(szPath, nFlag, nMode);
    set_fs(tFs);

    if(IS_ERR(ptFp))
    {
        *ptFd = NULL;
        return CAM_FS_FAIL;
    }
    else
    {
        *ptFd = (CamFsFd)ptFp;
        return CAM_FS_OK;
    }
#endif
}

CamFsRet_e CamFsClose(CamFsFd tFd)
{
#ifdef CAM_OS_RTK
    return CAM_FS_OK;
#elif defined(CAM_OS_LINUX_USER)
    return CAM_FS_FAIL;
#elif defined(CAM_OS_LINUX_KERNEL)
    struct file *ptFp = (struct file *)tFd;

    if (ptFp)
    {
        return (!filp_close(ptFp, NULL))? CAM_FS_OK : CAM_FS_FAIL;
    }
    else
    {
        return CAM_FS_FAIL;
    }
#endif
}

s32 CamFsRead(CamFsFd tFd, void *pBuf, u32 nCount)
{
#ifdef CAM_OS_RTK
    DrvSpinandProbe();

#if 1
    // Read NAND flash by block number
    DrvSpinand_ReadBlockPba((u8 *)pBuf, (u32)tFd, nCount);
#else
    if((tHal.u32Did == 0x21AA)&&(tHal.u8Mid == 0xEF))
    {
        (void)MDrv_SPINAND_LoadWB((u8 *)pBuf, nCount, (u32)tFd);
    }
    else /*others*/
    {
        (void)MDrv_SPINAND_LoadBL((u8 *)pBuf, nCount, TRUE, (u32)tFd);
    }
#endif

    return nCount;
#elif defined(CAM_OS_LINUX_USER)
    return 0;
#elif defined(CAM_OS_LINUX_KERNEL)
    struct file *ptFp = (struct file *)tFd;
    mm_segment_t tFs;
    loff_t tPos;
    s32 nRet;

    if (ptFp)
    {
        tFs = get_fs();
        set_fs(get_ds());
        tPos = ptFp->f_pos;
        nRet = vfs_read(ptFp, pBuf, nCount, &tPos);
        ptFp->f_pos = tPos;
        set_fs(tFs);
        return nRet;
    }
    else
    {
        return -1;
    }
#endif
}

s32 CamFsWrite(CamFsFd tFd, const void *pBuf, u32 nCount)
{
#ifdef CAM_OS_RTK
    return 0;
#elif defined(CAM_OS_LINUX_USER)
    return 0;
#elif defined(CAM_OS_LINUX_KERNEL)
    struct file *ptFp = (struct file *)tFd;
    mm_segment_t tFs;
    loff_t tPos;
    s32 nRet;

    if (ptFp)
    {
        tFs = get_fs();
        set_fs(get_ds());
        tPos = ptFp->f_pos;
        nRet = vfs_write(ptFp, pBuf, nCount, &tPos);
        ptFp->f_pos = tPos;
        set_fs(tFs);
        return nRet;
    }
    else
    {
        return -1;
    }
#endif
}
